#if NUNIT

#region References

using System;
using System.Collections;
using System.Text;

using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

using gov.va.med.vbecs.UnitTests;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.OpenLibrary.Messages
{
	[TestFixture]
	public class Messages_HL7AckMessage : BaseTest 
	{
		#region Variables & Setup

		private const string ACK_MESSAGE_TYPE = "ACK";
		private const string ACK_TRIGGER_EVENT = "A08";

		private static string _hl7Message;

		HL7AckMessage _ackMessage;

		static Messages_HL7AckMessage()
		{
			_hl7Message = string.Concat(
				"MSH^~|\\&^WBC^500^VBECS^500^20080520072932^^BTS~O31^WBC200805200729324027^P^2.5^^^AL^AL^", 
				CARRIAGE_RETURN,
				"PID^^^212236^^AAHPHAANY~ILUUHAA~^^19561101^M^^^^^^^^^^^450207089^^^^^^^^^^^^^^^^^^^^",
				CARRIAGE_RETURN,
				"PV1^^O^10W P-KC~~^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^", 
				CARRIAGE_RETURN,
				"ORC^SC^22179268~OR^^^^^^^^^^53342~ASKEW~DAVID^^^^^^^^^^^^^", 
				CARRIAGE_RETURN,
				"BPO^^~PLATELETS^^", 
				CARRIAGE_RETURN,
				"BTX^^f33825f3-fdad-44f0-8b28-624819f14ae1^12000^~A POS^^^^^115^^TX^^20080519172625^ASKEW,DAVID^USER,ONE^20080519172524^20080519172625^~Yes^STOPPING, REACTION, ,New Cough ,New Back Pain ,Nausea, with or without Vomiting ,IV Line Infiltrated ,Hypoxemia ,Hives, Urticaria (Rash) ,Headache, Severe ,Flushing ,Dyspnea (difficulty breathing) ,Chills (rigors) ,Bag Damaged ,Anxiety ,Other interruption reason ,Fever (>1C or 2F increase with temp >= 100F)", 
				CARRIAGE_RETURN,
				"NTE^^^MAIN COMMENT HERE^RE NTE^^^Testing Purposes^VA-BCR",
				CARRIAGE_RETURN );
		}

		[SetUp]
		public void SetUp()
		{
			UnitTestUtil.RunSQL( "UPDATE InterfaceControl SET InterfaceFacilityId = 'TEST'" );
			
			_ackMessage = new HL7AckMessage( BuildAckMessage() );
		}

		#endregion

		#region Constructors

		[Test]
		public void Ctor_Pass()
		{
			HL7AckMessage ackMessage = new HL7AckMessage( BuildAckMessage() );
			//
			Assert.IsTrue( ackMessage != null );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void Ctor_Fail()
		{
			HL7AckMessage ackMessage = new HL7AckMessage( null );
			//
			Assert.IsTrue( ackMessage != null );
		}

		#endregion

		#region Methods

		[Test]
		public void GetMessage_Pass()
		{
			Assert.IsNotNull( _ackMessage.GetMessage() );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetMessage_Fail()
		{
		}

		[Test]
		public void GetMessageControlID_Pass()
		{
			Assert.IsNotNull( _ackMessage.GetMessageControlID() );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetMessageControlID_Fail()
		{
		}

		[Test]
		public void GetMessageType_Pass()
		{
			Assert.IsNotNull( _ackMessage.GetMessageType() );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetMessageType_Fail()
		{
		}

		[Test]
		public void IsResponseRequired_Pass()
		{
			Assert.IsTrue( _ackMessage.IsResponseRequired() == true );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsResponseRequired_Fail()
		{
		}

		#endregion

		#region Properties

		[Test]
		public void MessageType_Pass()
		{
			Assert.IsNotNull( _ackMessage.MessageType );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void MessageType_Fail()
		{
		}

		[Test]
		public void Message_Pass()
		{
			Assert.IsNotNull( _ackMessage.Message );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void Message_Fail()
		{
		}

		[Test]
		public void MessageControlID_Pass()
		{
			Assert.IsNotNull( _ackMessage.MessageControlID );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void MessageControlID_Fail()
		{
		}

		[Test]
		public void MSH_Pass()
		{
			Assert.IsNotNull( _ackMessage.MSH );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void MSH_Fail()
		{
		}

		[Test]
		public void MSA_Pass()
		{
			Assert.IsNotNull( _ackMessage.MSA );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void MSA_Fail()
		{
		}

		#endregion

		#region Helper Methods

		/// <summary>
		/// 
		/// </summary>
		private string BuildAckMessage()
		{
			HL7Interface hl7Interface = new HL7Interface( "Patient_Update" );
			//
			StringBuilder ack = new StringBuilder();
			//
			ack.Append(BuildMSHSegment(hl7Interface));
			ack.Append("MSA");
			ack.Append(hl7Interface.FieldSeparator);
			ack.Append("AA");
			ack.Append(hl7Interface.FieldSeparator);
			ack.Append(HL7Utility.GetMessageControlID(_hl7Message));
			ack.Append(hl7Interface.FieldSeparator);
			ack.Append(CARRIAGE_RETURN);
			//
			return ack.ToString();
		}

		/// <summary>
		/// 
		/// </summary>
		private static string BuildMSHSegment(HL7Interface intParms)
		{
			StringBuilder _msh = new StringBuilder();
			//
			_msh.Append("MSH");										// MSH Segment ID
			_msh.Append(intParms.FieldSeparator);					// Field Separator
			_msh.Append(intParms.EncodingCharacters);				// Encoding Characters
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.VbecsApplicationId.Trim());		// Sending Application
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.VbecsFacilityId.Trim());			// Sending Facility
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.InterfaceApplicationId.Trim());	// Receiving Application
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.InterfaceFacilityId.Trim());		// Receiving Facility
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":",""));		// HL7 DateTime
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(ACK_MESSAGE_TYPE);							// ACK Message Type
			_msh.Append(intParms.EncodingCharacters[0]);
			_msh.Append(ACK_TRIGGER_EVENT);							// ACK Trigger Event
			_msh.Append(intParms.FieldSeparator);
			_msh.Append("VBECS" + System.DateTime.Now.ToString("yMMddhhssffff"));		// Message Control ID
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.ProcessingId);						// Processing ID
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.VersionId.Trim());					// HL7 Version ID
			_msh.Append(CARRIAGE_RETURN);
			//
			return _msh.ToString();
		}

		#endregion
	}
}
#endif